/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved.
*/

#ifndef UUID_6a3daf08_c579_41b2_83b2_e70827c54142
#define UUID_6a3daf08_c579_41b2_83b2_e70827c54142

/**
 * \file vnccdbsdk.h
 *
 * This is the main header file for the VNC CDB and SBP SDK. Applications using
 * the SDK should include only this file.
 *
 * This file includes \ref vncint.h. Applications including vnccdbsdk.h must
 * therefore define either VNC_USE_DEFAULT_INT_TYPEDEFS or VNC_USE_STDINT_H,
 * as described in vncint.h.
 *
 * \see vnccdb.h, vnccdbtypes.h, vncsbp.h, vncsbpserialize.h, vncsbptypes.h
 */

/**
 * \mainpage VNC CDB and SBP SDK
 *
 * \section section_contents Contents
 *
 *  -# \ref section_overview
 *  -# \ref section_usage
 *  -# \ref section_references
 *  -# \ref section_legal
 *
 * \section section_overview Overview
 *
 * VNC CDB and SDK implements MirrorLink's Common Data Bus protocol
 * as defined in \ref ml_cdb "[ML-CDB]", and MirrorLink's Service Binary
 * Protocol, as defined in \ref ml_sbp "[ML-SBP]".
 *
 * The SDK can act as a CDB source or CDB sink, and as a MirrorLink client or
 * MirrorLink server. In any of these cases, the application must be able to
 * provide the SDK with the following information:
 *
 *  - the MirrorLink version implemented by the MirrorLink server
 *  - the URL of the server-side CDB endpoint
 *
 * The MirrorLink server version can be found in the root device XML of the
 * MirrorLink server.
 *
 * If the calling application is the MirrorLink client, then the URL of the
 * server-side CDB endpoint can be obtained from the
 * TmApplicationServerService, in its response to a LaunchApplication action
 * (see \ref ml_app "[ML-APP]"). MirrorLink UPnP is implemented by VNC
 * Discovery SDK.
 *
 * If the calling application is the MirrorLink server, then the URL of the
 * server-side CDB endpoint is at the application's discretion. The SDK will
 * listen for an incoming connection on the network interface and TCP port
 * number indicated in the URL.
 *
 * \section section_usage Basic usage
 *
 * \subsection subsection_usage_compiling Compiling your application
 *
 * To use the SDK, your application must first define exactly one of the
 * preprocessor macros VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS.  These
 * macros govern the definitions of the vnc_*_t integer types defined by
 * vncint.h.
 *
 *  - If your build environment provides the ISO C99 header stdint.h, or you
 *    can author an equivalent yourself, you should define VNC_USE_STDINT_H.
 *    This is the recommended approach for defining the SDK's integer types.
 *  - If you examine vncint.h and find that the types used when you define
 *    VNC_USE_DEFAULT_INT_TYPEDEFS are correct for your platform, then you may
 *    wish to define VNC_USE_DEFAULT_INT_TYPEDEFS instead of VNC_USE_STDINT_H.
 *    Additionally, if you are targeting a 64-bit platform with a compiler
 *    other than Microsoft Visual C++ or gcc, then you may have to define
 *    VNC_FORCE_64_BIT in order to define vnc_intptr_t and vnc_uintptr_t
 *    correctly.  See vncint.h itself for further information.
 *
 * The types defined by VNC_USE_DEFAULT_INT_TYPEDEFS are known to be correct
 * for Microsoft Visual C++ 2005.
 *
 * After defining either VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS, your
 * application should include vnccdbsdk.h.  You are encouraged to use the
 * VNCVerifyIntegerTypes() macro to verify that the integer types have been
 * defined correctly.
 *
 * \subsection subsection_usage_startup Initializing the SDK
 *
 * -# Load the SDK DLL (necessary only if you did not link against it).
 *    - on Windows, call LoadLibrary()
 *    - on Linux, call dlopen()
 * -# Locate the SDK entry point, VNCCDBSDKInitialize().
 *    - on Windows, call GetProcAddress()
 *    - on Linux, call dlsym()
 * -# Call VNCCDBSDKInitialize().  This populates a VNCCDBSDK structure with
 *   pointers to the rest of the SDK's 'methods'.
 *
 * Typically, your application will call VNCCDBSDKInitialize() at start-up.
 * You should not call VNCCDBSDKInitialize() more than once.
 *
 * \subsection subsection_usage_creating Creating a CDB endpoint
 *
 * Call VNCCDBEndpointCreate() to create a CDB endpoint. You may reuse the same CDB
 * endpoint for a succession of CDB sessions that require a similar
 * configuration. There is no need to create a new CDB endpoint for each
 * session (although the application is free to do so if it wishes, and there
 * is no limit on the number of CDB endpoints that may exist simultaneously).
 *
 * When you create a CDB endpoint, you supply two structures to the SDK:
 *
 *  - VNCCDBEndpointDescriptor - Contains information about the endpoint, such
 *    whether the endpoint is a MirrorLink client or a MirrorLink server, as the
 *    server endpoint's URL, the desired protocol version, and an opaque
 *    'context' data pointer for use by the application.
 *  - VNCCDBEndpointCallbacks - Pointers to various callbacks that provide your
 *    application with information about CDB events. It is recommended that you
 *    implement VNCCDBEndpointLogCallback(), so that you are given detailed
 *    diagnostic information.
 *
 * \subsection subsection_usage_licensing Licensing a CDB endpoint
 *
 * In order to use the SDK, the application must always provide each CDB
 * endpoint instance with at least one license. Use VNCCDBEndpointAddLicense()
 * to do so. Contact your RealVNC sales representative to obtain a license.
 *
 * \subsection subsection_usage_sources Defining SBP and CDB sources
 *
 * If your CDB endpoint will host any CDB services (i.e. if it will act as a
 * CDB source), then the next step is to define each such service using
 * VNCCDBEndpointAddSBPSource() or VNCCDBEndpointAddCDBSource(). Each of these
 * APIs defines a service to be hosted by the CDB endpoint and provides the SDK
 * with callbacks that it will invoke when the service is started, stopped or
 * otherwise interacted with by the peer endpoint.
 *
 * VNCCDBEndpointAddSBPSource() is a high-level API that defines a service that
 * will be accessed only via SBP. It is recommended that the application uses
 * this API for all SBP services.
 *
 * VNCCDBEndpointAddCDBSource() is a lower-level API that defines a service
 * that will be accessed via CDB without the use of SBP. It is suitable for use
 * by CDB services that do not use SBP, but instead use CDB together with some
 * other data representation and data access protocol.
 *
 * A single CDB endpoint instance may act as both a CDB source <i>and</i> a CDB
 * sink simultaneously. All source services must be defined before the endpoint
 * is started.
 *
 * \subsection subsection_usage_starting Starting the CDB endpoint
 *
 * To start the CDB endpoint, call VNCCDBEndpointStart(). This API creates a
 * new thread of execution (see \ref subsection_usage_threading) which attempts
 * to establish the connection to the peer endpoint. In the case of a client
 * endpoint, an outward TCP connection attempt is made; in the case of a
 * server endpoint, the endpoint binds to the address and port in the endpoint
 * URL and then listens for an incoming TCP connection.
 *
 * Once the connection is established, the CDB endpoint then automatically
 * sends a ServicesRequest CDB message. When the peer endpoint responds with a
 * ServicesSupported CDB message, the SDK invokes the
 * VNCCDBEndpointServicesSupportedCallback() associated with the endpoint.
 *
 * \subsection subsection_usage_sinks Defining SBP and CDB sinks
 *
 * During or after the VNCCDBEndpointServicesSupportedCallback(), the
 * application may register sinks for any CDB services advertised by the peer.
 * Use VNCCDBEndpointAddSBPSink() or VNCCDBEndpointAddCDBSink() to register a
 * sink.
 *
 * VNCCDBEndpointAddSBPSink() is a high-level API that defines a sink for a
 * service that will be accessed only via SBP. It is recommended that the
 * application uses this API for all SBP services.
 *
 * VNCCDBEndpointAddCDBSink() is a lower-level API that defines a sink for a
 * service that will be accessed via CDB without the use of SBP. It is suitable
 * for use with CDB services that do not use SBP, but instead use CDB together
 * with some other data representation and data access protocol.
 *
 * CDB sinks are automatically unregistered by the SDK when the CDB session
 * terminates. This is because the list of services hosted by the peer CDB
 * endpoint may change between sessions (or even during a session). Even if the
 * list of services hosted by the peer does not change, the service IDs of
 * those services are not guaranteed to remain constant between sessions. For
 * this reason, it is necessary for the application to register all of its
 * sinks in every session. The VNCCDBEndpointServicesSupportedCallback()
 * provides a good opportunity to do this.
 *
 * A single CDB endpoint instance may act as both a CDB source <i>and</i> a CDB
 * sink simultaneously. Service sinks may not be defined until the
 * VNCCDBEndpointServicesSupportedCallback() is invoked.
 *
 * \subsection subsection_usage_destroying Stopping the CDB endpoint
 *
 * To intentionally terminate a CDB session, call VNCCDBEndpointStop(). This
 * begins the process of closing the session and disconnecting from the peer
 * endpoint (or, if the session has not yet been established, stops trying to
 * establish the session).
 *
 * The SDK always reports ::VNCCDBEndpointStatusStopping via the
 * VNCCDBEndpointStatusCallback() immediately before the CDB thread exits. This
 * call will always happen, whether the application calls VNCCDBEndpointStop()
 * or the session terminates for some other reason.
 *
 * VNCCDBEndpointStop() does not return until the CDB thread has exited.
 *
 * \subsection subsection_usage_destroying Destroying the CDB endpoint
 *
 * To destroy a CDB endpoint, call VNCCDBEndpointDestroy().
 *
 * If the CDB thread is running, then VNCCDBEndpointDestroy() makes an implicit
 * call to VNCCDBEndpointStop(). Therefore, VNCCDBEndpointDestroy() also will
 * not return until the CDB thread has exited.
 *
 * \subsection subsection_usage_uninitializing Uninitializing the SDK
 *
 * After all CDB endpoints have been destroyed, but before exiting, the
 * application should call VNCCDBSDKUninitialize().
 *
 * \subsection subsection_usage_threading Threading
 *
 * Calling VNCCDBEndpointStart() creates a new thread of execution. This thread
 * is known as the CDB thread. It is this thread that established the
 * connection to the CDB peer and manages the CDB protocol.
 *
 * <b>All callbacks from the SDK to the application are made from the CDB
 * thread.</b> The first callback to be made from the CDB thread will 
 * always be VNCCDBEndpointThreadStartedCallback(), and last will always be
 * VNCCDBEndpointThreadStoppedCallback().
 * 
 * It is the application's responsibility to ensure that these callbacks are 
 * handled in a thread-safe manner. The only exception to this is
 * the VNCCDBEndpointLogCallback(), which may be invoked by any thread that
 * calls SDK API, as well as by the CDB thread. However, the SDK guarantees
 * that it will never enter the VNCCDBEndpointLogCallback() from more than one
 * thread at a time.
 *
 * With the exception of VNCCDBEndpointCreate() and VNCCDBEndpointDestroy(),
 * all SDK API calls are safe for use by any application thread at any time.
 * (The SDK marshalls the majority of API calls onto a thread-safe queue and
 * then processes them within the CDB thread.)
 *
 * \section section_references References
 *
 *  -# \anchor ml_app [ML-APP] CCC-TS-024 UPnP Application Server Service,
 *     versions 1.1 and 1.2
 *  -# \anchor ml_cdb [ML-CDB] CCC-TS-016 Common Data Bus, versions 1.1 and 1.2
 *  -# \anchor ml_sbp [ML-SBP] CCC-TS-018 Service Binary Protocol, versions 1.1
 *     and 1.2
 *
 * \section section_legal Legal information
 *
 * Copyright &copy; 2002-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * Details of and copyright notices for third-party software that is used by
 * the VNC CDB and SBP SDK can be found in the file Acknowledgements.txt in the
 * SDK distribution.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * MirrorLink is a registered trademark of Car Connectivity Consortium in the
 * U.S. and in other countries.
 *
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCVerifyIntegerTypes, VNCCDBSDK, VNCCDBSDKInitialize,
 * VNCCDBSDKUninitialize, VNCCDBEndpointCreate
 */

#include "vnccall.h"
#include "vncint.h"
#include "vnccdbtypes.h"
#include "vnccdb.h"
#include "vncsbptypes.h"
#include "vncsbpserialize.h"
#include "vncsbp.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Returns a symbolic name corresponding to an SDK error code.
 *
 * For example, if error is VNCCDBErrorInvalidParameter, then the returned
 * string is "InvalidParameter". Such strings may be useful as
 * internationalization keys or in log entries.
 *
 * The strings returned by VNCCDBGetErrorName() are brief and are not
 * internationalized. They are therefore not suitable for display to an end
 * user.
 * 
 * \param error The error code.
 * \return A short UTF-8 string corresponding to the error code. The memory
 * pointed to is owned by the SDK; do not free it. If the error code is not
 * recognized by the SDK, then the return value is NULL.
 */
typedef const char *VNCCALL
VNCCDBGetErrorName(VNCCDBError error);

/**
 * \brief Uninitializes the SDK.
 *
 * The application should call this function after destroying all CDB endpoints
 * but before unloading the SDK.
 *
 * \see VNCCDBSDKInitialize(), VNCCDBEndpointDestroy()
 */
typedef void VNCCALL
VNCCDBSDKUninitialize(void);

/**
 * \brief Contains the SDK's version and the addresses of the SDK's methods.
 * 
 * \see VNCCDBSDKInitialize()
 */
typedef struct
{
    /** Major component of the SDK's version number. */
    vnc_int32_t versionMajor;
    /** Minor component of the SDK's version number. */
    vnc_int32_t versionMinor;
    /** Patch component of the SDK's version number. */
    vnc_int32_t versionPatch;
    /** Build number component of the SDK's version number. */
    vnc_int32_t versionBuild;

    /** Uninitializes the SDK. */
    VNCCDBSDKUninitialize *vncCDBSDKUninitialize;
    /** Returns a symbolic name corresponding to an SDK error code. */
    VNCCDBGetErrorName *vncCDBGetErrorName;

    /** Creates a CDB endpoint. */
    VNCCDBEndpointCreate *vncCDBEndpointCreate;
    /** Destroys a CDB endpoint. */
    VNCCDBEndpointDestroy *vncCDBEndpointDestroy;
    /** Adds a license to a CDB endpoint. */
    VNCCDBEndpointAddLicense *vncCDBEndpointAddLicense;
    /** Adds a CDB source service to a CDB endpoint. */
    VNCCDBEndpointAddCDBSource *vncCDBEndpointAddCDBSource;
    /** Adds a CDB sink for one of the peer's services to a CDB endpoint. */
    VNCCDBEndpointAddCDBSink *vncCDBEndpointAddCDBSink;
    /** Starts a CDB endpoint. */
    VNCCDBEndpointStart *vncCDBEndpointStart;
    /** Stops a CDB endpoint. */
    VNCCDBEndpointStop *vncCDBEndpointStop;
    /** Sends a CDB StartService message. */
    VNCCDBEndpointSendStartService *vncCDBEndpointSendStartService;
    /** Sends a CDB StopService message. */
    VNCCDBEndpointSendStopService *vncCDBEndpointSendStopService;
    /** Sends a CDB ServicePayload message. */
    VNCCDBEndpointSendServicePayload *vncCDBEndpointSendServicePayload;
    /** Sends a CDB ServiceResponse message. */
    VNCCDBEndpointSendServiceResponse *vncCDBEndpointSendServiceResponse;
    /** Returns the context pointer associated with a CDB endpoint. */
    VNCCDBEndpointGetContext *vncCDBEndpointGetContext;
    /** Adds a SBP source service to a CDB endpoint. */
    VNCCDBEndpointAddSBPSource *vncCDBEndpointAddSBPSource;
    /** Adds a SBP sink for one of the peer's services to a CDB endpoint. */
    VNCCDBEndpointAddSBPSink *vncCDBEndpointAddSBPSink;

    /** Implements the hash function that generates SBP UIDs for strings. */
    VNCSBPHash *vncSBPHash;

    /** Creates and returns a new ::VNCSBPSerialize instance. */
    VNCSBPSerializeCreate *vncSBPSerializeCreate;
    /** Destroys a ::VNCSBPSerialize instance. */
    VNCSBPSerializeDestroy *vncSBPSerializeDestroy;
    /** Finalizes the serialized payload stored in a ::VNCSBPSerialize */
    VNCSBPSerializeFinalize *vncSBPSerializeFinalize;
    /** Appends a Boolean to a serialized SBP payload. */
    VNCSBPSerializeBoolean *vncSBPSerializeBoolean;
    /** Appends a signed byte to a serialized SBP payload. */
    VNCSBPSerializeByte *vncSBPSerializeByte;
    /** Appends a signed 16-bit integer to a serialized SBP payload. */
    VNCSBPSerializeShort *vncSBPSerializeShort;
    /** Appends a signed 32-bit integer to a serialized SBP payload. */
    VNCSBPSerializeInt *vncSBPSerializeInt;
    /** Appends a signed 64-bit integer to a serialized SBP payload. */
    VNCSBPSerializeLong *vncSBPSerializeLong;
    /**
     * Appends a single-precision IEEE 754-1985 floating point number to a
     * serialized SBP payload.
     */
    VNCSBPSerializeFloat *vncSBPSerializeFloat;
    /**
     * Appends a double-precision IEEE 754-1985 floating point number to a
     * serialized SBP payload.
     */
    VNCSBPSerializeDouble *vncSBPSerializeDouble;
    /** Appends an array of bytes to a serialized SBP payload. */
    VNCSBPSerializeBytes *vncSBPSerializeBytes;
    /** Appends a text string to a serialized SBP payload. */
    VNCSBPSerializeString *vncSBPSerializeString;
    /** Begins serialization of an SBP packet body. */
    VNCSBPSerializeBeginPacket *vncSBPSerializeBeginPacket;
    /** Begins serialization of a STRUCTURE. */
    VNCSBPSerializeBeginStructure *vncSBPSerializeBeginStructure;
    /** Begins serialization of an ARRAY or a STRUCTURE_ARRAY. */
    VNCSBPSerializeBeginArray *vncSBPSerializeBeginArray;
    /** Ends serialization of an ARRAY, a STRUCTURE or a STRUCTURE_ARRAY. */
    VNCSBPSerializeEnd *vncSBPSerializeEnd;

    /** Creates a new ::VNCSBPDeserialize instance. */
    VNCSBPDeserializeCreate *vncSBPDeserializeCreate;
    /** Destroys a ::VNCSBPDeserialize instance. */
    VNCSBPDeserializeDestroy *vncSBPDeserializeDestroy;
    /** Obtains context pointed from a ::VNCSBPDeserialize instance. */
    VNCSBPDeserializeGetContext *vncSBPDeserializeGetContext;
    /** Deserializes a single (potentially compound) SBP field. */
    VNCSBPDeserializeExecute *vncSBPDeserializeExecute;
    /** Deserializes an entire SBP packet body. */
    VNCSBPDeserializeExecutePacket *vncSBPDeserializeExecutePacket;

    /** Obtain the state of an object. */
    VNCSBPSinkSendGetRequest *vncSBPSinkSendGetRequest;
    /** Change the state of an object. */
    VNCSBPSinkSendSetRequest *vncSBPSinkSendSetRequest;
    /** Subscribe to updates for an object. */
    VNCSBPSinkSendSubscribeRequest *vncSBPSinkSendSubscribeRequest;
    /** Cancel a previous request. */
    VNCSBPSinkSendCancelRequest *vncSBPSinkSendCancelRequest;
    /** Send an authentication response. */
    VNCSBPSinkSendServiceAuthResponse *vncSBPSinkSendServiceAuthResponse;

    /** Send a response to a request for the state of an object. */
    VNCSBPSourceSendGetResponse *vncSBPSourceSendGetResponse;
    /** Send a response to a request to set the state of an object. */
    VNCSBPSourceSendSetResponse *vncSBPSourceSendSetResponse;
    /** Send a response to a request to subscribe to an object. */
    VNCSBPSourceSendSubscribeResponse *vncSBPSourceSendSubscribeResponse;
    /** Send an authentication challenge. */
    VNCSBPSourceSendServiceAuthChallenge *vncSBPSourceSendServiceAuthChallenge;

    /** Check if a service is still responding. */
    VNCSBPSinkSendServiceAliveRequest *vncSBPSinkSendServiceAliveRequest;

} VNCCDBSDK;

/**
 * \brief The type of VNCCDBSDKInitialize().
 *
 * If you are dynamically loading the SDK at runtime, you should use this
 * typedef to declare your function pointer.
 *
 * \see VNCCDBSDKInitialize()
 */
typedef VNCCDBError VNCCALL
VNCCDBSDKInitializeType(VNCCDBSDK *pSDK, size_t sdkSize);

/**
 * \brief The entry point to the SDK.
 *
 * This function is the sole entry point exported by the SDK.  Before you can
 * call any other of the SDK's functions, you must call this function to obtain
 * pointers to their implementations.
 *
 * Before unloading the SDK, you should call VNCCDBSDKUninitialize().
 *
 * \param pSDK Pointer to the VNCCDBSDK structure to be initialized.
 * \param sdkSize Size of the VNCCDBSDK structure to be initialized.
 *
 * \retval ::VNCCDBErrorNone The SDK was initialized successfully.
 * \retval ::VNCCDBErrorInvalidParameter sdkSize is not valid.
 *
 * \see ::VNCCDBSDK, VNCCDBSDKUninitialize(), ::VNCCDBSDKInitializeType,
 * VNCVerifyIntegerTypes()
 */
VNCDLLIMPORT VNCCDBError VNCCALL
VNCCDBSDKInitialize(VNCCDBSDK *pSDK, size_t sdkSize);

#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_6a3daf08_c579_41b2_83b2_e70827c54142) */
